﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Numerics;
using System.Security.Cryptography;
using System.Text;
using System.Threading.Tasks;
using TMPro;
using UnityEngine;
using UnityEngine.PlayerLoop;
using UnityEngine.Tilemaps;

//WagerRaceMission represents and manages a wager race mission - a bet oriented time trial
public class WagerRaceMission : Mission
{
    //WagerRaceDifficultyData is a container struct for difficulty specific parameters
    public class WagerRaceDifficultyData
    {
        public int Time { get; private set; }
        public int MinWager { get; private set; }
        public int MaxWager { get; private set; }
        public float OddsMultiplier { get; private set; }
        public Vector3Int Destination { get; private set; }

        public WagerRaceDifficultyData(int time, int minWager, int maxWager, float oddsMultiplier, Vector3Int destination)
        {
            Time = time;
            MinWager = minWager;
            MaxWager = maxWager;
            OddsMultiplier = oddsMultiplier;
            Destination = destination;
        }
    }

    public int Wager { get; private set; } = 0;
    public int Returns { get; private set; } = 0;

    public Vector3Int Destination { get; private set; }
    public Dictionary<Constants.MissionDifficulties, WagerRaceDifficultyData> DifficultyData { get; private set; } = new Dictionary<Constants.MissionDifficulties, WagerRaceDifficultyData>();

    private float _TimeRemaining = 0;

    //UI Elements
    private GameObject _BaseUIObject;
    private MissionTimerController _MissionTimer;
    private TMP_Text _DestinationText;
    private TMP_Text _WagerText;
    private TMP_Text _ReturnsText;

    public WagerRaceMission(Vector3Int missionSpawnPos) : base(missionSpawnPos)
    {

    }

    public override bool Generate()
    {
        try
        {
            //Let's get the possible destinations
            List<Vector3Int> possibleDestinations = GameController.Instance.LSystem.Drawer.RoadPositions;
            possibleDestinations.Shuffle();

            //And get the data for difficulties
            DifficultyData[Constants.MissionDifficulties.Easy] = GenerateDataForDifficulty(ConfigurationManager.Instance.Missions.WagerRace.Easy, ConfigurationManager.Instance.Missions.Core.Time.Easy, possibleDestinations);
            DifficultyData[Constants.MissionDifficulties.Medium] = GenerateDataForDifficulty(ConfigurationManager.Instance.Missions.WagerRace.Medium, ConfigurationManager.Instance.Missions.Core.Time.Medium, possibleDestinations);
            DifficultyData[Constants.MissionDifficulties.Hard] = GenerateDataForDifficulty(ConfigurationManager.Instance.Missions.WagerRace.Hard, ConfigurationManager.Instance.Missions.Core.Time.Hard, possibleDestinations);
            IsGenerated = true;
            return true;
        }

        catch (Exception ex)
        {
            Debug.LogError("ERROR: Caught an exception when generating wager race mission, returning false. The exception is: " + ex);
            return false;
        }
    }

    public override void StartMission(Constants.MissionDifficulties difficulty)
    {
        //Store the difficulty, update the minimap
        SelectedDifficulty = difficulty;
        Destination = DifficultyData[SelectedDifficulty].Destination;
        MinimapManager.Instance.AddIcon(DifficultyData[SelectedDifficulty].Destination, Constants.WagerRaceDestinationIconName, ColoursManager.Instance.Colours["MissionDestinationIcon"].Colour, ConfigurationManager.Instance.Missions.WagerRace.DestinationIconRenderSize, 0, Resources.Load<Sprite>("Minimap/Icons/minimapDestinationIcon"));

        //Update the UI
        _TimeRemaining = DifficultyData[SelectedDifficulty].Time;
        _BaseUIObject = GameManager.Instance.MissionsUIGameObject.FindChild("WagerRace");
        _MissionTimer = _BaseUIObject.FindChild("HeaderFlyout").FindChild("Timer").GetComponent<MissionTimerController>();
        _DestinationText = _BaseUIObject.FindChild("DestinationText").GetComponent<TMP_Text>();

        //Let's get the street the destination is on to display in the UI
        if (StreetsManager.Instance.Streets.ContainsKey(Destination))
        {
            _DestinationText.SetText("Get to <color=yellow>" + StreetsManager.Instance.Streets[Destination] + "</color> before time runs out!");
            _DestinationText.gameObject.SetActive(true);
        }

        else
        {
            _DestinationText.gameObject.SetActive(false);
        }

        //Also update the wager info
        _WagerText = _BaseUIObject.FindChild("WagerFlyout").FindChild("WagerText").GetComponent<TMP_Text>();
        _WagerText.text = "WAGER: $" + Wager;
        _ReturnsText = _BaseUIObject.FindChild("ReturnsFlyout").FindChild("ReturnsText").GetComponent<TMP_Text>();
        _ReturnsText.text = "RETURNS: $" + Returns;

        //Subtract the wager amount from the player
        GameManager.Instance.IncrementMoney(-Wager);

        //Add warning to timer from 10s downwards
        for (int i = 10000; i >= 0; i -= 1000)
        {
            _MissionTimer.AddTime(i);
        }

        //Start the timer and the mission
        _MissionTimer.StartTimer(DifficultyData[SelectedDifficulty].Time);
        UpdateTimeUI();
        _BaseUIObject.SetActive(true);
        _IsMissionStarted = true;
    }

    public override void EndMission(bool passed)
    {
        //Stop the timer, tidy up the minimap
        _MissionTimer.StopTimer();
        MinimapManager.Instance.RemoveIcon(Constants.WagerRaceDestinationIconName);

        //And reward the player if they passed
        if (passed)
        {
            GameManager.Instance.IncrementMoney(Returns);
        }

        //Mission is ended
        _BaseUIObject.SetActive(false);
        _IsMissionStarted = false;
    }

    /// <summary>
    /// Sets the wager and return values that have been calculated
    /// </summary>
    /// <param name="wager">The amount to wager</param>
    /// <param name="returns">The amount to be returned if won</param>
    public void SetWagerValues(int wager, int returns)
    {
        Wager = wager;
        Returns = returns;
    }

    public override string GetRewardText()
    {
        if (SelectedDifficulty != Constants.MissionDifficulties.Sentinel)
        {
            return "+$" + Returns;
        }

        return "";
    }

    public override void SpawnMinimapIcon(string iconID)
    {
        MinimapManager.Instance.AddIcon(MissionSpawnPosition, iconID, ColoursManager.Instance.Colours["WagerRaceMissionIcon"].Colour, ConfigurationManager.Instance.Minimap.MissionIconRenderSize, 0, Resources.Load<Sprite>("Minimap/Icons/minimapWagerRaceIcon"));
    }

    public override void Update()
    {
        if (_IsMissionStarted)
        {
            //Decrement the time left, update the UI
            _TimeRemaining -= (Time.deltaTime * 1000.0f);
            UpdateTimeUI();

            if (_TimeRemaining <= 0.0f)
            {
                //Out of time, mission failed
                MissionsManager.Instance.EndMission(false);
            }

            else
            {
                Vector3Int tilePos = GameManager.Instance.RoadsTilemap.WorldToCell(GameManager.Instance.PlayerCarGameObject.transform.position);

                if (tilePos == Destination)
                {
                    //The player is at the destination, mission passed!
                    MissionsManager.Instance.EndMission(true);
                }
            }
        }
    }

    private WagerRaceDifficultyData GenerateDataForDifficulty(WagerRaceDifficultyConfiguration missionConfig, MissionsTimeDifficultyConfiguration timeConfig, List<Vector3Int> possibleDestinations)
    {
        //Generate a random destination
        Randomizer.Regenerate();
        int index = Randomizer.RNG.Next(0, possibleDestinations.Count);
        Vector3Int generatedDestination = possibleDestinations[index];

        //Generate an amount of time
        int generatedTime = GenerateTimeForDifficulty(timeConfig, MissionSpawnPosition, generatedDestination);
        generatedTime = Convert.ToInt32(Math.Ceiling(generatedTime * (1.0f / ConfigurationManager.Instance.Missions.Core.Time.RoundFactor)) * ConfigurationManager.Instance.Missions.Core.Time.RoundFactor);

        Randomizer.Regenerate();
        int oddsMultiplierInteger = Randomizer.RNG.Next(Convert.ToInt32(missionConfig.OddsMultiplierLowerRange * 100.0f), Convert.ToInt32(missionConfig.OddsMultiplierUpperRange * 100.0f));
        float generatedOddsMultiplier = Convert.ToSingle(Math.Round(((oddsMultiplierInteger / 100.0f) / ConfigurationManager.Instance.Missions.WagerRace.OddsMultiplierRoundFactor), 0) * ConfigurationManager.Instance.Missions.WagerRace.OddsMultiplierRoundFactor);

        return new WagerRaceDifficultyData(generatedTime, missionConfig.MinWagerCap, missionConfig.MaxWagerCap, generatedOddsMultiplier, generatedDestination);
    }

    private void UpdateTimeUI()
    {
        _MissionTimer.UpdateTimer(_TimeRemaining);
    }
}
